// Exécute le script une fois le DOM entièrement chargé
document.addEventListener('DOMContentLoaded', function () {

  // Sélectionne tous les formulaires d'ajout au panier
  document.querySelectorAll('.add-to-cart-form').forEach(form => {

    // Récupère les combinaisons encodées en JSON depuis le dataset du formulaire
    const dataset = form.dataset.combinations;
    if (!dataset) return;

    let combinations = {};
    try {
      combinations = JSON.parse(dataset); // Convertit le JSON en objet JS
    } catch (e) {
      console.error('combinations JSON parse error', e);
      return;
    }

    // Normalise les combinaisons en un tableau d'objets clairs pour manipulation
    const combos = Object.keys(combinations).map(k => {
      const c = combinations[k];
      return {
        id_combination: Number(k),
        size_id: c.size_id ? Number(c.size_id) : null,
        color_id: c.color_id ? Number(c.color_id) : null,
        quantity: Number(c.quantity || 0),
      };
    }).filter(c => c.quantity > 0); // Garde uniquement les combinaisons avec du stock

    // Récupère tous les <select> d’attributs (groupes taille, couleur, etc.)
    const selects = Array.from(form.querySelectorAll('select[name^="group["]'));

    /**
     * Fonction utilitaire pour activer/désactiver une option de select
     */
    function toggleOption(opt, enable) {
      opt.disabled = !enable;
      opt.style.opacity = enable ? 1 : 0.6;
      opt.classList.toggle('unavailable', !enable);
    }

    /**
     * Met à jour la disponibilité des options selon les sélections utilisateur
     * et tente d’identifier la combinaison exacte sélectionnée
     */
    function updateAvailabilityAndCombination() {

      // Récupère les valeurs actuellement sélectionnées dans chaque select
      const selected = {};
      selects.forEach(s => {
        const groupId = s.getAttribute('data-attr-group');
        selected[groupId] = s.value ? Number(s.value) : null;
      });

      // Pour chaque select, on parcourt ses options et on vérifie si elles sont valides
      selects.forEach(s => {
        const groupId = s.getAttribute('data-attr-group');

        s.querySelectorAll('option').forEach(opt => {

          if (!opt.value) {
            // L’option vide (“Choisir”) reste toujours activée
            toggleOption(opt, true);
            return;
          }

          const optId = Number(opt.value);

          // Vérifie si au moins une combinaison valide correspond à cette option
          const valid = combos.some(c => {

            // Vérifie la cohérence avec les autres sélections
            for (const otherSelect of selects) {
              const otherGroupId = otherSelect.getAttribute('data-attr-group');
              const selVal = selected[otherGroupId];
              if (!selVal) continue; // Aucune contrainte pour ce groupe

              // Si la combinaison ne contient pas l’attribut sélectionné, elle n’est pas valide
              if (c.size_id !== null && c.size_id === selVal) continue;
              if (c.color_id !== null && c.color_id === selVal) continue;
              return false;
            }

            // Vérifie que cette option fait partie de la combinaison (taille ou couleur)
            return (c.size_id === optId) || (c.color_id === optId);
          });

          // Active ou désactive l’option selon la validité trouvée
          toggleOption(opt, valid);
        });
      });

      // Vérifie si toutes les sélections sont complètes (tous les groupes remplis)
      const allSelectedFilled = selects.every(s => s.value && s.value !== '');
      const idProductAttrField = form.querySelector('.id_product_attribute_field');

      if (allSelectedFilled) {
        // Recherche la combinaison exacte correspondant à toutes les sélections
        const selectedIds = selects.map(s => Number(s.value));
        const found = combos.find(c => {
          // Chaque attribut sélectionné doit correspondre à la combinaison
          return selectedIds.every(selId => (c.size_id === selId) || (c.color_id === selId));
        });

        // Si trouvée, on met à jour le champ caché id_product_attribute
        if (found) {
          idProductAttrField.value = found.id_combination;
        } else {
          // Sinon, on le vide (aucune combinaison exacte correspondante)
          idProductAttrField.value = '';
        }
      } else {
        // Si toutes les sélections ne sont pas faites, on vide le champ
        // Le contrôleur du panier utilisera les valeurs de group[...] pour résoudre
        idProductAttrField.value = '';
      }
    }

    // Ajoute les écouteurs d’événements sur les sélecteurs d’attributs
    selects.forEach(s => s.addEventListener('change', updateAvailabilityAndCombination));

    // Exécute une première fois la mise à jour à l’initialisation
    updateAvailabilityAndCombination();
  });
});