// Exécute le script une fois que le DOM est entièrement chargé
document.addEventListener('DOMContentLoaded', function () {
  
  // ================================
  // 1. GESTION DU MENU DÉROULANT DE LA NAVIGATION STORE
  // ================================

  const storeNav = document.querySelector('.store-nav'); // Conteneur du menu de sélection de magasin

  if (storeNav) {
    const dropdown = storeNav.querySelector('.store-nav .dropdown-menu'); // Menu déroulant à afficher/masquer

    // Toggle du menu au clic sur la navigation
    storeNav.addEventListener('click', function () {
      dropdown.classList.add('show'); // Ajoute ou retire la classe "show"
    });

    // Ferme le menu si on clique en dehors de la navigation
    document.addEventListener('click', function (e) {
      if (!storeNav.contains(e.target)) {
        dropdown.classList.remove('show'); // Cache le menu si clic à l'extérieur
      }
    });
  }

  // ================================
  // 2. DÉFINITION DES CONSTANTES ET ÉLÉMENTS DU POPUP
  // ================================

  const COOKIE_NAME = 'popup_store'; // Nom du cookie à créer
  const HOURS_VALID = 12; // Durée de validité du cookie (en heures)
  const SHOW_DELAY = 2000; // délai en millisecondes (ici 1.5 secondes)

  const popup   = document.querySelector('#popup-store'); // Élément du popup
  const overlay = document.querySelector('.overlay'); // Fond d’écran flouté derrière le popup
  const closeBtn = document.querySelector('.close-popup-store'); // Bouton pour fermer le popup

  // Récupération des paramètres de l'URL
  const urlParams = new URLSearchParams(window.location.search);
  const storeSelected = urlParams.get('store_selected'); // Vérifie si le paramètre existe

  // ================================
  // 3. FONCTIONS UTILITAIRES POUR LES COOKIES
  // ================================

  function setCookie(name, value, hours) {
    const d = new Date();
    d.setTime(d.getTime() + hours * 60 * 60 * 1000); // Ajoute X heures
    document.cookie = encodeURIComponent(name) + '=' + encodeURIComponent(value) +
      ';expires=' + d.toUTCString() + ';path=/;SameSite=Lax'; // Définition du cookie
  }

  function getCookie(name) {
    const match = document.cookie.match(new RegExp('(^| )' + name + '=([^;]+)'));
    return match ? decodeURIComponent(match[2]) : null; // Retourne la valeur ou null
  }

  // ================================
  // 4. TRAITEMENT SI UN STORE EST SÉLECTIONNÉ
  // ================================

  if (storeSelected) {
    // Si le paramètre "store_selected" est présent → on crée le cookie
    setCookie(COOKIE_NAME, '1', HOURS_VALID);

    // Nettoie l’URL en supprimant le paramètre après traitement
    urlParams.delete('store_selected');
    const cleanUrl = window.location.pathname + (urlParams.toString() ? '?' + urlParams.toString() : '');
    window.history.replaceState({}, document.title, cleanUrl);
  }

  // ================================
  // 5. AFFICHAGE OU NON DU POPUP SELON LE COOKIE
  // ================================

  if (!getCookie(COOKIE_NAME)) {
    // Si le cookie n’existe pas, on affiche le popup
    popup.style.display = 'block';
    overlay.style.display = 'block';

    // Petite pause pour les animations CSS
    setTimeout(function () {
      document.body.classList.add('no-scroll'); // Empêche de scroller derrière le popup
      popup.classList.add('show');              // Ajoute la classe CSS animée
      overlay.classList.add('show');            // Idem pour l’overlay
    },SHOW_DELAY);
  } else {
    // Si le cookie existe, on masque tout
    popup.style.display = 'none';
    overlay.style.display = 'none';
  }

  // ================================
  // 6. FERMETURE DU POPUP VIA LE BOUTON
  // ================================

  if (closeBtn) {
    closeBtn.addEventListener('click', function () {
      popup.classList.remove('show');
      overlay.classList.remove('show');

      // Attend la fin de l’animation avant de cacher complètement
      setTimeout(function () {
        document.body.classList.remove('no-scroll');
        popup.style.display = 'none';
        overlay.style.display = 'none';
      }, 300); // Durée de l'animation en ms

      // On définit le cookie pour ne plus afficher le popup pendant la durée choisie
      setCookie(COOKIE_NAME, '1', HOURS_VALID);
    });
  }

// ================================
// 7. GESTION DU CLIC SUR UN MAGASIN
// ================================

// Liens dans la popup ou liste classique
const popupLinks = document.querySelectorAll('.store-item a');

// Liens dans le menu déroulant de navigation
const dropdownLinks = document.querySelectorAll('.store-nav .dropdown-menu a');

// Fonction commune pour gérer le clic et ajouter le paramètre + redirection
function handleStoreLinkClick(link) {
  link.addEventListener('click', (e) => {
    e.preventDefault(); // Bloque la redirection par défaut

    const url = new URL(link.href); // On crée un objet URL à partir du href
    url.searchParams.set('store_selected', '1'); // Ajoute le paramètre pour la logique cookie

    window.location.href = url.toString(); // Redirige vers l'URL avec le paramètre
  });
}

// Ajoute le comportement aux deux types de liens
popupLinks.forEach(handleStoreLinkClick);
dropdownLinks.forEach(handleStoreLinkClick);

});