<?php 

namespace PrestaShop\Module\PolarisPrestaTheme\Controllers\Admin;

use Configuration;
use Link;
use PolarisPrestaTheme;
use PrestaShop\Module\PolarisPrestaTheme\Classes\AutoUpdater;
use PrestaShop\Module\PolarisPrestaTheme\Forms\ModuleConfGeneralType;
use PrestaShop\PrestaShop\Adapter\SymfonyContainer;
use Symfony\Component\HttpFoundation\Request;

class ConfigurationAdminController extends \PrestaShopBundle\Controller\Admin\PrestaShopAdminController
{
    /**
     * lien vers le module
     */
    protected PolarisPrestaTheme $module;

    /**
     * Initialisation
     * 
     */
    public function  __construct() 
    {
        $this->module = \Module::getInstanceByName(strtolower('PolarisPrestaTheme'));    
    }

    /**
     * Retourne les variables passées en paramètre enrichies des variables par défaut
     * 
     * @param array $vars Les variables à enrichir
     */
    protected function addDefaultVars(array $vars) : array
    {
        $vars['module_name'] = $this->module->name;

        if (!isset($vars['layoutHeaderToolbarBtn']))
            $vars['layoutHeaderToolbarBtn'] = $this->getAdminLayoutToolBar();
        
        return $vars;
    }    

    /**
     * On enrechit la configuration de la barre d'outils
     * 
     * @return array
     */
    protected function getAdminLayoutToolBar() : array 
    {
        return [];
    }

    /**
     * Convertit du markdown en HTML, en mode simplifié (on a pas besoin de plus)
     */
    private function markdownToHtml(string $markdown): string
    {
        // On commence l'analyse au premier titre (première caractère #)
        // avec un bon vieux strpos
        $pos = strpos($markdown, '#');
        if ($pos !== false)
            $markdown = substr($markdown, $pos);        

        // Convertir les titres
        $markdown = preg_replace('/^#### (.*)$/m', '<h4>$1</h4>', $markdown);
        $markdown = preg_replace('/^### (.*)$/m', '<h3>$1</h3>', $markdown);
        $markdown = preg_replace('/^## (.*)$/m', '<h2>$1</h2>', $markdown);
        $markdown = preg_replace('/^# (.*)$/m', '<h1>$1</h1>', $markdown);

        // Convertir le gras et l'italique
        $markdown = preg_replace('/\*\*(.*?)\*\*/', '<strong>$1</strong>', $markdown); // **texte**
        $markdown = preg_replace('/\*(.*?)\*/', '<em>$1</em>', $markdown);           // *texte*
        // ```code``` => <pre></pre>
        $markdown = preg_replace('/```(.*?)```/s', '<pre>$1</pre>', $markdown);
        // ```code``` => <code></code>
        $markdown = preg_replace('/`(.*?)`/s', '<code>$1</code>', $markdown);

        // Convertir les listes
//        $markdown = preg_replace('/^- (.*)$/m', '<li>$1</li>', $markdown);
//        $markdown = preg_replace('/^\* (.*)$/m', '<li>$1</li>', $markdown);

        // Convertir les listes Markdown en HTML
        $lines = explode("\n", $markdown);
        $html = '';
        $inList = false;

        foreach ($lines as $line) {
            $trimmed = trim($line);

            // Détecter une liste
            if (preg_match('/^\* (.+)$/', $trimmed, $matches)) {
                if (!$inList) {
                    $html .= "<ul>\n";
                    $inList = true;
                }
                $html .= "<li>{$matches[1]}</li>\n";
            } else {
                // Si on sort d'une liste, fermer la balise <ul>
                if ($inList) {
                    $html .= "</ul>\n";
                    $inList = false;
                }
                // Ajouter les autres lignes normalement
                $html .= $line . "\n";
            }
        }

        // S'assurer de fermer une liste ouverte en fin de fichier
        if ($inList) {
            $html .= "</ul>\n";
        }

        $markdown = $html;

        // Convertir les liens
        $markdown = preg_replace('/\[(.*?)\]\((.*?)\)/', '<a href="$2">$1</a>', $markdown); // [texte](url)

        // Convertir les sauts de ligne, que quand ils sont doublés
        $markdown = preg_replace('/\n\n/', '<br>', $markdown);
        
        return $markdown;
    }  

    /** 
     * Rendu du Changelog
     */
    public function changelogDisplayAction() {

        $see = '';
        $doc = @file_get_contents($this->module->getLocalPath() . 'CHANGELOG.md');
        if (!$doc)
            $doc = "Le fichier CHANGELOG.md n'a pas pu être chargé.";        

        $refreshed = false;
        if (isset($_REQUEST['refresh_update']) && $_REQUEST['refresh_update'])
        {
            $refreshed = true;

            $updater = new AutoUpdater($this->module);
            $updater->searchUpdate();
        }
        

        return $this->render('@Modules/' . $this->module->name . '/views/templates/admin/showdoc.html.twig', $this->addDefaultVars([
            'version' => $this->module->installed_version,
            'version_update' => $this->module->version != $this->module->installed_version ? $this->module->version : null,
            'refreshed' => $refreshed,
            'see' => $see,
            'doc_html' => $this->markdownToHtml($doc),
            'alert_url' => SymfonyContainer::getInstance()->get('router')->generate('prestatheme_security_alert'),
            'employee'=> \Context::getContext()->employee->id_profile
        ]));
    }

    /**
     * Gestion du formulaire de configuration des informations
     */
    public function generalConfigurationAction(Request $request)
    {    
        $form = $this->createForm(ModuleConfGeneralType::class);
        
        $form->handleRequest($request);

        if($form->isSubmitted() && $form->isValid())
        {
            $cfg = $this->module->loadCacheCfg();
            $cfg['ENABLED'] = $form->get('cache_active')->getData() ? true : false;
            $cfg['INVALID_PRODUCT_CACHE'] = $form->get('cache_auto_invalidate')->getData() ? true : false;
            $cfg['CACHE_MAX_AGE'] = $form->get('cache_duration')->getData();

            // Sauvegarde de la configuration
            Configuration::updateValue(PolarisPrestaTheme::MODULE_PREFIX . "CACHE_CFG", json_encode($cfg));

       

            $this->addFlash('success', 'Configuration sauvegardée !');

            // On recharge le formulaire pour éviter de garder des données non corrigée à l'affichage
            $form = $this->createForm(ModuleConfGeneralType::class);
        }

        return $this->render('@Modules/' . $this->module->name . '/views/templates/admin/module_conf.html.twig', $this->addDefaultVars([
            'form' => $form->createView(),
        ]));
    }
}