<?php

namespace PrestaShop\Module\PolarisPrestaTheme\Controllers\Admin;

use Symfony\Component\HttpFoundation\Request;
use PrestaShop\Module\PolarisPrestaTheme\Config\ConfigTheme;
use PrestaShop\Module\PolarisPrestaTheme\Forms\{MenuParentType, MenuChildType, MenuMediaType};
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * AdminMenuController
 * 
 * Contrôleur pour la configuration des menus personnalisés.
 * Permet de gérer les menus parents et leurs sous-menus (colonnes).
 */
class AdminMenuController extends AdminFormController
{
    /**
     * ConfigurationMenuParentAction
     * 
     * Permet au client de configurer un menu principal (parent)
     * 
     * @param Request $request
     * @param int $id L'ID du menu parent
     * @param string $action Action à effectuer (ajouter, modifier, supprimer)
     */
    public function ConfigurationMenuParentAction(Request $request, int $id, $action)
    {
        return $this->CreateConfigurationForm(
            $request,
            MenuParentType::class,                                         // Type de formulaire pour les catégories
            '\PrestaShop\Module\PolarisPrestaTheme\Config\ConfigMenuParent',     // Classe de configuration
            $id,                                                           // ID de la catégorie
            $action,                                                       // Action à effectuer
            'configMenuParent',                                            // Nom du tableau
            'admin_presta_show_parent_menu',                               // Redirection après traitement
            false,                                                         // L'image doit être uploadée
            'Configuration du premier niveau de votre menu'                                        // Titre de la page
        );
    }

    /**
     * ConfigurationMenuChildAction
     * 
     * Permet au client d'ajouter des sous-catégories à un menu parent
     * 
     * @param Request $request
     * @param int $id ID de la sous-catégorie
     * @param string $action Action à effectuer
     * @param int $parent_id ID du menu parent
     */
    public function ConfigurationMenuChildAction(Request $request, int $id, $action, $parent_id)
    {
        return $this->CreateConfigurationForm(
            $request,
            MenuChildType::class,                                          // Type de formulaire 
            '\PrestaShop\Module\PolarisPrestaTheme\Config\ConfigMenuChild',      // Classe de configuration
            $id,                                                           // ID
            $action,                                                       // Action à effectuer
            'configMenuChild',                                             // Nom du tableau
            'admin_presta_show_child_menu',                                // Redirection après traitement
            false,                                                         // L'image doit être uploadée
            'Ajouter les sous-catégories',                                 // Titre de la page
            $parent_id                                                     // ID de la catégorie parente
        );
    }

    /**
     * ShowMenuParentAction
     * 
     * Affiche les menus parents configurés
     */
    public function ShowMenuParentAction()
    {
        // Chargement de la configuration
        $config = ConfigTheme::load();

        // Retourne la vue avec les menus parents configurés
        return $this->render('@Modules/polarisprestatheme/views/templates/admin/show_menu_parent.html.twig', [
            'menus' => $config->configMenuParent
        ]);
    }

    /**
     * ShowMenuChildAction
     * 
     * Affiche les sous-menus (colonnes) associés à un menu parent
     * 
     * @param int $parent_id ID du menu parent
     */
    public function ShowMenuChildAction($parent_id)
    { 
        // Chargement de la configuration
        $config = ConfigTheme::load();
        $columns = [];

        // On parcourt les menus parents pour trouver celui correspondant
        foreach ($config->configMenuParent as $menuParent) {
            if ((int) $menuParent->menu['id'] === (int) $parent_id) {
                $columns = $menuParent->menu['columns'] ?? [];
                break;
            }
        }

        // Retourne la vue avec les colonnes associées au parent
        return $this->render('@Modules/polarisprestatheme/views/templates/admin/show_menu_child.html.twig', [
            'menus'     => $columns,
            'parent_id' => $parent_id,
        ]);
    }


    /**
     * ConfigurationMenuMediaAction
     * 
     * Permet au client d'ajouter un média dans un sous-menu
     * 
     * @param Request $request
     * @param int $id ID du média
     * @param string $action Action à effectuer (ajouter, modifier, supprimer)
     * @param int $parent_id ID de la catégorie parente
     */
    public function ConfigurationMenuMediaAction(Request $request, int $id, $action, $parent_id)
    {
        return $this->CreateConfigurationForm(
            $request,
            MenuMediaType::class,                                          // Type de formulaire 
            '\PrestaShop\Module\PolarisPrestaTheme\Config\ConfigMenuMedia',      // Classe de configuration
            $id,                                                           // ID
            $action,                                                       // Action à effectuer
            'configMenuMedia',                                             // Nom du tableau
            'admin_presta_show_child_menu',                                // Redirection après traitement
            true,                                                         // L'image doit être uploadée
            'Ajouter un media dans votre sous-menu',                                 // Titre de la page
             $parent_id                                                     // ID de la catégorie parente
        );
    }


    /**
     * ReorderParentMenuAction
     * 
     * Permet de réordonner les menus parents via drag & drop
     * 
     * @param Request $request
     */
    public function ReorderParentMenuAction(Request $request)
    {
        // Charge la configuration JSON actuelle
        $config = ConfigTheme::load();

        // Données envoyées depuis le JavaScript (tableau de IDs)
        $data = json_decode($request->getContent(), true);

        // On récupère le tableau des IDs dans l'ordre souhaité
        $order = $data['order'] ?? [];

        // On crée un nouveau tableau de menus parents triés
        $sorted = [];

        // Pour chaque ID dans le nouvel ordre
        foreach ($order as $id) {
            // On cherche dans les menus existants celui qui correspond à l'ID
            foreach ($config->configMenuParent as $item) {
                if ((int)$item->menu['id'] === (int)$id) {
                    $sorted[] = $item;
                    break;
                }
            }
        }

        // On remplace le tableau complet par le nouveau trié
        $config->configMenuParent = $sorted;

        // On sauvegarde le fichier
        $config->save();

        // Réponse AJAX OK
        return new JsonResponse(['success' => true]);
    }


    /**
     * ReorderChildMenuAction
     * 
     * Permet de réordonner les sous-menus (colonnes) d'un menu parent via drag & drop
     * 
     * @param Request $request
     */
    public function ReorderChildMenuAction(Request $request)
    {
        // On charge la config depuis le fichier JSON
        $config = ConfigTheme::load();

        // On récupère les données envoyées en POST JSON
        $data = json_decode($request->getContent(), true);

        
        // Récupère l'ID du menu parent (le menu principal)
        $parentId = (int) $data['id_parent'];

        // Récupère le nouvel ordre des colonnes sous forme d'un tableau d'index : [2, 1] ou [1, 2]
        $newOrder = $data['order'] ?? [];

        // Ce tableau contiendra le nouveau configMenuParent complet (réécrit proprement)
        $newConfigMenuParent = [];

        // On parcourt tous les menus parents actuels
        foreach ($config->configMenuParent as $menuParent) {
            
            // Si ce n’est pas le menu concerné, on le conserve tel quel
            if ((int)$menuParent->menu['id'] !== $parentId) {
                $newConfigMenuParent[] = $menuParent;
                continue;
            }

            // Ici on est sur le bon menu parent : on veut réordonner ses colonnes

            // On récupère les colonnes actuelles
            $oldColumns = $menuParent->menu['columns'];

           

            // Ce tableau contiendra les colonnes dans le bon ordre
            $reordered = [];

            // Pour chaque index reçu depuis le JS (ex: 2, 1)
            foreach ($newOrder as $columnIndex) {
                $columnIndexStr = (string) $columnIndex; // Les clés sont des strings dans le JSON

                // On ajoute la colonne correspondante dans le nouvel ordre
                if (isset($oldColumns[$columnIndexStr])) {
                    $reordered[$columnIndexStr] = $oldColumns[$columnIndexStr];
                }
            }

            // On remplace les anciennes colonnes par les colonnes réordonnées
            $menuParent->menu['columns'] = $reordered;

            // On ajoute ce menu modifié dans la nouvelle config
            $newConfigMenuParent[] = $menuParent;
        }

        // On remplace l’ancien tableau complet par le nouveau
        $config->configMenuParent = $newConfigMenuParent;

        // On sauvegarde dans le fichier JSON
        $config->save();

        // On renvoie une réponse JSON de succès
        return new JsonResponse(['success' => true]);
    }
}
