<?php
namespace PrestaShop\Module\PolarisPrestaConnector;

use PolarisPrestaConnector;

class Upgrade {

    /**
     * Complète la liste des couleurs
     */
    public static function upgradeColorLists(PolarisPrestaConnector $module)
    {
        include __DIR__ . '/colors.php';
        
        $colorMap = $module->getCfg()->get(SyncConfiguration::CFG_COLOR_MAP, []);

        if (!isset($colorMap["strict"]))
            $colorMap["strict"] = [];

        // Pour chaque couleur existante dans la liste des couleurs
        // on va regarder si elle est défini dans la map des couleurs
        // si ce n'est pas le cas ou qu'elle est vide, alors on map
        // la couleur sur elle-même
        $changed = false;
        foreach ($colorDefinitions as $color => $value) {
            if (!isset($colorMap["strict"][$color]) || empty($colorMap["strict"][$color]))
            {                
                $colorMap["strict"][$color] = $value;
                $changed = true;
            }
        }

        // On fait la même chose pour les couleurs primaires
        foreach ($primaryDefinition as $color => $def) {
            if (!isset($colorMap["primary"][$color]) || empty($colorMap["primary"][$color]))
            {
                $colorMap["primary"][$color] = $def;
                $changed = true;
            }
        }

        // Sauvegarde de la nouvelle map
        if ($changed)
            $module->getCfg()->set(SyncConfiguration::CFG_COLOR_MAP, $colorMap);            
    }

    /**
     * Déclenche la mise à jour de la base de données du module
     */
    public function upgradeModule(PolarisPrestaConnector $module, string $version)
    {
        $doing = "init";
        try {
            // A partir d'ici, capturer toute les sorties PHP
            ob_start();

            // On scanne le répertoire de ce fichier à la recherche de tous les fichiers upgrade-{$version}.php possible
            // on trie les fichiers par ordre croissant de version qu'il représente
            $files = glob(__DIR__ . '/upgrade-*.php');
            usort($files, function($a, $b) {
                $a = explode('-', basename($a));
                $b = explode('-', basename($b));
                return version_compare($a[1], $b[1]);
            });
            
            // Pour chaque fichier dont la version est supérieure à la version actuelle du module ($version)
            // on l'inclut et on appelle la méthode upgrade
            $db = \Db::getInstance();

            foreach ($files as $file) {
                $fileVersion = explode('-', basename($file, '.php'))[1];
                $doing = $fileVersion;
                if (version_compare($fileVersion, $version) > 0) {                
                    require_once $file;
                    // La classe est le nom de la version avec _ à la place des .
                    $classname = '\\PrestaShop\\Module\\PolarisPrestaConnector\\Upgrade_' . str_replace('.', '_', $fileVersion);
                    $upgrade = new $classname();
                    $upgrade->upgrade($module, $db);

                    // Puis on met à jour la version du module dans notre configuration
                    \Configuration::updateValue($module->BridgePrefix . 'prestaconnector_version', $fileVersion);
                }
            }

            // Finalisation
            $doing = $module->installed_version;

            // Et maintenant, on applique la configuration par défaut
            $module->getCfg()->applyDefaults();

            // On réenregistre les hooks
            $module->registerHooks(true);

            // On réenregistre les onglets
            $module->registerTabs();

            // On réenregistre les couleurs
            self::upgradeColorLists($module);

            // Enfin, on fixe la version actuelle
            \Configuration::updateValue($module->BridgePrefix . 'prestaconnector_version', $module->installed_version);
        }
        catch (\Exception $e) {
            throw new \Exception("Erreur à la mise à jour v".$doing." : ".$e->getMessage());
        }
        finally {
            // On vide le buffer de sortie dans une variable
            $output = ob_get_clean();

            if (!empty(trim($output))) 
            {
                // On affiche le contenu du buffer dans les journaux Prestashop
                // En tant qu'erreur
                \PrestaShopLogger::addLog("Erreur lors de la mise à jour de la DB : ".$output, 4, null, $module->name);
            }
        }        
    }
}