/**
 * Déclenche une opération asynchronise depuis un bouton, devant le mettre à jour !
 * 
 * @param {*} elt 
 * @param {*} url 
 */
async function polarisprestaconnector_ajaxButtonReactionCall(elt, url) {

    // On commence par remplacer le texte du bouton par un loader
    // On cache l'élément (display : none)
    const before = elt.style.display;
    elt.style.display = 'none';

    // On ajoute un loader qu'on affiche juste à côté de l'élément
    const spinner = document.createElement('div');
    spinner.className = 'spinner';
    spinner.style.width = '18px';
    spinner.style.height = '18px';

    const loader = document.createElement('div');
    loader.className = elt.className + ' disabled';
    loader.appendChild(spinner);

    let result = false;

    // Insertion du loader juste après l'élement
    elt.parentNode.insertBefore(loader, elt.nextSibling); 

    try
    {
        const baseUrl = $('body').data('base-url');
        const token = $('body').data('token');

        // Appel ajax pour synchroniser le produit
        const response = await fetch(`${baseUrl}/modules/polarisprestaconnector/${url}&_token=${token}`, {
            method: 'GET',
            JSON: true,
            headers: {
                'Content-Type': 'application/json'
            }
        });        

        if (!response.ok)
            throw new Error('Erreur HTTP : ' + response);

        // Normalement, la réponse devrait contenir un JSON avec une propriété 'success' à vrai
        // Si ce n'est pas le cas, on lève une erreur
        const data = await response.json();
        if (data.success)
        {
            result = true;
        }
        else
        {
            console.error(data);
            // On montre l'erreur avec les fonctions JS prévu par PrestaShop
            showErrorMessage(data.message);            
        }
    }
    catch (e)
    {
        console.error(e);
        showErrorMessage('Erreur : ' + e.message);
    }
    finally
    {
        // On montre l'élément !
        elt.style.display = before;
        // On supprime le loader
        loader.remove();
    }

    return result;
}

/**
 * Déclenche une resynchronisation globale
 */
async function polarisprestaconnector_resyncAll() {
    console.debug('Resynchronisation totale demandée !');

    // Si un bouton avec un id 'page-header-desc-configuration-resync' existe, on le désactive et on change son nom
    const elt = document.getElementById('page-header-desc-configuration-resync');
    
    if (await polarisprestaconnector_ajaxButtonReactionCall(elt, 'resync_all?'))
        elt.innerHTML = 'Resynchro. demandée !';
}

/**
 * Lance une synchronisation
 *  
 */
async function polarisprestaconnector_run(cronGuardToken) {
    console.debug('Lancement de la synchronisation !');

    const elt = document.getElementById('page-header-desc-configuration-run');    

    elt.className = elt.className + ' disabled';
    // On change l'icône du bouton pour un spinner : 
    // pour ça, on cherche dans les fils la balise "i" et on change son contenu
    const i = elt.querySelector('i');
    if (i !== null)
        i.innerHTML = 'sync';

    // Appel ajax vers la tâche cron, sans rien d'autre !
    await fetch(`${baseDir}module/polarisprestaconnector/cron?guard=${cronGuardToken}`, {
        method: 'GET',
        JSON: true,
        headers: {
            'Content-Type': 'application/json'
        }
    });
}

/**
 * Met à jour l'environnement
 */
async function polarisprestaconnector_pullenv() {
    console.debug('Téléchargement de l\'environnement depuis le backoffice !');

    const elt = document.getElementById('page-header-desc-configuration-pullenv');    

    await polarisprestaconnector_ajaxButtonReactionCall(elt, 'configuration/pullenv?');
}

/**
 * Obtient la progression de la synchronisation en cours ou null si aucune
 * 
 */
async function polarisprestaconnector_getSyncProgress() {

    try
    {
        const baseUrl = $('body').data('base-url');
        const token = $('body').data('token');

        // Appel ajax pour synchroniser le produit
        const response = await fetch(`${baseUrl}/modules/polarisprestaconnector/syncprogress?&_token=${token}`, {
            method: 'GET',
            JSON: true,
            headers: {
                'Content-Type': 'application/json'
            }
        });        

        if (!response.ok)
            throw new Error('Erreur HTTP : ' + response);

        // Normalement, la réponse devrait contenir un JSON avec une propriété 'success' à vrai
        // Si ce n'est pas le cas, on lève une erreur
        const data = await response.json();
        if (data.success)
        {
            return data.progress;
        }
        else
        {
            console.error(data);
            // On montre l'erreur avec les fonctions JS prévu par PrestaShop
            showErrorMessage(data.message);            
        }
    }
    catch (e)
    {
        console.error(e);
        showErrorMessage('Erreur : ' + e.message);
    }

    return null;
}

/**
 * Vérifie la progression de la synchronisation en cours à intervalles réguliers
 */
async function polarisprestaconnector_adminSyncProgressCheck() {

    const elt = document.getElementById('page-header-desc-configuration-run');    
    if (!elt)
        return;

    const result = await polarisprestaconnector_getSyncProgress();
    if (result !== null && result.progress && result.progress.running)
    {
        // On ajoute disabled aux classes CSS s'il n'y est pas déjà 
        if (!elt.className.includes('disabled'))
        {
            elt.className = elt.className + ' disabled';
            // On change l'icône du bouton pour un spinner : 
            // pour ça, on cherche dans les fils la balise "i" et on change son contenu
            const i = elt.querySelector('i');
            if (i !== null)
                i.innerHTML = 'sync';
        }
    }
    else
    {
        // On retire "disabled" des classes CSS
        if (elt.className.includes('disabled'))
        {
            elt.className = elt.className.replace('disabled', '');
            // On change l'icône du bouton pour une icône play
            const i = elt.querySelector('i');
            if (i !== null)
                i.innerHTML = 'play_circle';
        }
    }
}

/**
 * Affiche la progression de la synchronisation en cours
 */
async function polarisprestaconnector_dashboardSyncProgressShow(progress)
{
    const elt = document.getElementById('polarisprestaconnector_dashboard_progress');    
    if (!elt)
        return;

    if (progress)
    {
        if (progress.running)
        {            
            let html = '<p><strong>Synchronisation en cours</strong>';
            
            // On marque la durée de la synchronisation, du style : démarrée depuis 24minutes et 30 secondes...
            const startTimestamp = progress.startTime * 1000;
            const nowTimestamp = Date.now();
            const diffDate = new Date(nowTimestamp - startTimestamp);
            const diffStr = (diffDate.getUTCHours() * 60  + diffDate.getUTCMinutes()) + 'mins ' + diffDate.getUTCSeconds() + 's';
            html += ', démarrée depuis ' + diffStr;
            html += '</p>';

            html += '<ul>';
            // foreach pour parcourir les clés de l'objet progress.steps
            for (const key in progress.steps)
            {
                const step = progress.steps[key];
                html += '<li';
                if (step.n > 0 && step.d == step.n)
                    html += ' style="list-style-type: \'✓ \'; text-decoration: line-through;"';
                else
                    html += ' class="blinking-text"';
                html += '>';
                html += step.m;
                html += '</li>';
            }

            html += '</ul>';

            //const perc = progress.stepNb > 0 ? Math.round(progress.stepDone * 100 / progress.stepNb) : 0;

            
            /*html += `<div class="progress"><div class="progress-bar progress-bar-striped progress-bar-animated" role="progressbar" 
                        style="width: ${perc}%"></div></div>`;*/                        

            elt.innerHTML = html;
        }
        else
        {
            if (progress.endTime)
            {            
                const endTimestamp = progress.endTime * 1000;
                const nowTimestamp = Date.now();
                const ellapsedTime = (nowTimestamp - endTimestamp) / 1000;

                let diffStr = '';            
                if (ellapsedTime < 60)
                    diffStr = 'moins d\'une minute';
                else if (ellapsedTime < 3600)
                    diffStr = Math.floor(ellapsedTime / 60) + ' minutes';
                else if (ellapsedTime < 86400 * 3)
                    diffStr = Math.floor(ellapsedTime / 3600) + ' heures';
                else
                    diffStr = Math.floor(ellapsedTime / 86400) + ' jours';

                elt.innerHTML = '<p>Dernière synchronisation il y a ' + diffStr + '.</p>';
            } 
            else
            {
                elt.innerHTML = '<p>Pas encore synchronisé.</p>';
            }

            if (progress.lastErrorMsg)
            {
                // Tronquer le message d'erreur à 100 caractères
                if (progress.lastErrorMsg.length > 120)
                    progress.lastErrorMsg = progress.lastErrorMsg.substring(0, 120) + '…';
                elt.innerHTML = '<div class="sync-error"><p>' + progress.lastErrorMsg + '</p></div>' + elt.innerHTML;
            }
        }
    }
    else
    {
        elt.innerHTML = '<p>Impossible d\'obtenir l\'état de la synchronisation</p>';
    }
}

/**
 * Déclenche la synchronisation d'un produit spécifique
 * @param {*} e évènement déclencheur
 * @param {*} id_product 
 * 
 */
async function polarisprestaconnector_syncProduct(e, id_product) {    
    
    console.debug('Synchronisation du produit ' + id_product);
    const elt = e.currentTarget;    

    if (await polarisprestaconnector_ajaxButtonReactionCall(elt, `sync_product?id_product=${id_product}`))
        document.location.reload();
}

/**
 * Déclenche la synchronisation d'une commande spécifique
 * @param {*} elt élement déclencheur
 * @param {*} id_product 
 * 
 */
async function polarisprestaconnector_exportOrder(elt, order_reference) {    
    
    console.debug('Exportation de la commande ' + order_reference);

    if (await polarisprestaconnector_ajaxButtonReactionCall(elt, `export_order?order_reference=${order_reference}`))
        document.location.reload();
}

/**
 * Déclenche la synchronisation d'un produit spécifique
 * @param {*} e évènement déclencheur
 * @param {*} id_product 
 * 
 */
async function polarisprestaconnector_toggleLockProduct(e, id_product) {    
    
    console.debug('Lock/Unlock produit ' + id_product);
    const elt = e.currentTarget;    
    const action = elt.innerHTML.includes('lock_open') ? 'lock' : 'unlock';

    if (await polarisprestaconnector_ajaxButtonReactionCall(elt, `lock_product?id_product=${id_product}&action=${action}`))
        document.location.reload();
}
