<?php
namespace PrestaShop\Module\LCVPrestaTheme\Classes;
use PrestaShopLogger;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use PrestaShop\Module\LCVPrestaTheme\Config\ConfigTheme;
use \Imagick;


/**
 * Ensemble d'utilitaires pour la gestion des fichiers et répertoires liés au thème PrestaShop.
 */
class ThemeTools
{
    /** 
     * removeDirectory 
     *
     * Supprime le dossier $path et tout son contenu.
     * 
     * @param string $path Dossier à supprimer
     * @throws Exception en cas d'erreur lors de la suppression
     */
    public static function removeDirectory(string $path): void
    {
        if (!is_dir($path)) {
            PrestaShopLogger::addLog("ℹ️ Le répertoire n'existe pas : $path", 1);
            return;
        }
    
        $handle = opendir($path);
        if (!$handle) {
            throw new \Exception("Impossible d’ouvrir le répertoire : $path");
        }
    
        try {
            while (($entry = readdir($handle)) !== false) {
                if ($entry === '.' || $entry === '..') {
                    continue;
                }
    
                $file = $path . DIRECTORY_SEPARATOR . $entry;
    
                if (is_dir($file)) {
                    self::removeDirectory($file);
                } else {
                    if (!unlink($file)) {
                        throw new \Exception("Impossible de supprimer le fichier : $file");
                    }
                }
            }
        } finally {
            closedir($handle);
    
            if (file_exists($path) && !rmdir($path)) {
                throw new \Exception("Impossible de supprimer le répertoire : $path");
            }
        }
    }

    /** 
     * copyTheme
     *
     * copie le dossier $source et tout son contenu vers sa $destination
     * 
     * @param string $soure fichier a copier
     * @param string $destination route ou copier le fichier
     * @throws Exception en cas d'erreur lors de la copie
     */
    public static function copyTheme($source, $destination)
    {
        if (!is_dir($source)) {
            PrestaShopLogger::addLog("❌ Le dossier source n'existe pas : $source", 3);
            return false;
        }
    
        if (!file_exists($destination)) {
            @mkdir($destination, 0755, true);
        }
    
        $dir = opendir($source);
        if (!$dir) {
            PrestaShopLogger::addLog("❌ Impossible d'ouvrir le dossier source : $source", 3);
            return false;
        }
    
        while (($file = readdir($dir)) !== false) {
            if ($file === '.' || $file === '..') {
                continue;
            }
    
            $srcPath = $source . '/' . $file;
            $destPath = $destination . '/' . $file;
    
            if (is_dir($srcPath)) {
                if (!self::copyTheme($srcPath, $destPath)) {
                    closedir($dir);
                    return false;
                }
            } else {
                if (!copy($srcPath, $destPath)) {
                    PrestaShopLogger::addLog("❌ Erreur lors de la copie de $srcPath vers $destPath", 3);
                    closedir($dir);
                    return false;
                }
            }
        }
    
        closedir($dir);
        return true;
    }

     
    /**
     * uploadImage
     *
     * Upload une image Symfony (UploadedFile), la convertit en WebP
     * et la stocke dans le dossier spécifié ($uploadDir).
     *
     * @param UploadedFile $file Fichier image uploadé via Symfony
     * @param string $uploadDir Dossier cible pour l’enregistrement
     * @return string URL publique de l’image WebP générée
     * @throws \Exception En cas d’erreur d’extension, d’upload ou de conversion
     */
    public static function uploadImage(UploadedFile $file, string $uploadDir): string
    {
    // Extensions autorisées
        $_registration_allowed_extensions = ['png', 'jpg', 'jpeg', 'webp'];

        // Extension du fichier
        $image_extension = $file->guessExtension();
        if (!in_array(strtolower($image_extension), $_registration_allowed_extensions)) {
            throw new \Exception("Extension de fichier non prise en charge.");
        }

        // Nom original (sans l’extension)
        $original_name = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
        $image_name = $original_name . '.' . $image_extension;
        $finalPath = $uploadDir . DIRECTORY_SEPARATOR . $image_name;

        // Éviter les collisions de nom
        $counter = 1;
        while (file_exists($finalPath)) {
            $image_name = $original_name . '_' . $counter . '.' . $image_extension;
            $finalPath = $uploadDir . DIRECTORY_SEPARATOR . $image_name;
            $counter++;
        }

        // Déplacement du fichier depuis le cache Symfony vers le répertoire cible
        $file->move($uploadDir, $image_name);

        // Vérification de l’existence réelle
        if (!file_exists($finalPath)) {
            throw new \Exception("Le fichier uploadé n'existe pas : " . $finalPath);
        }

        // Utilisation de ImageMagick pour convertir en WebP
        if (class_exists('Imagick')) {
            try {
                // Création d'un objet Imagick pour l'image téléchargée
                $imagick = new Imagick($finalPath);
                
                // Conversion au format WebP
                $imagick->setImageFormat('webp');
                
                // Définir une qualité de compression (facultatif, de 0 à 100)
                $imagick->setImageCompressionQuality(100);
                
                // Définir le chemin de l'image WebP
                $webpPath = $uploadDir . DIRECTORY_SEPARATOR . $original_name . '.webp';
                
                // Sauvegarde de l'image convertie en WebP
                $imagick->writeImage($webpPath);
                
                // Libération des ressources
                $imagick->clear();
                
                // Suppression de l’image originale
                unlink($finalPath);
                
            } catch (\Exception $e) {
                throw new \Exception("Erreur de conversion avec Imagick: " . $e->getMessage());
            }
        } else {
            throw new \Exception("Imagick n'est pas installé sur le serveur.");
        }

        // Retourne l’URL publique de l’image WebP
        return __PS_BASE_URI__ . 'themes/presta_theme/assets/img/' . $original_name . '.webp';

    }




    /**
     * updateCssFile
     * 
     * Met à jour le fichier var.css avec les nouvelles variables de configuration
     * 
     */
    public static function updateCssFile(){

         $config = ConfigTheme::load();

        // Chemin vers le fichier var.css
        $fileVarCss = _PS_ROOT_DIR_ . DIRECTORY_SEPARATOR . 'themes' . DIRECTORY_SEPARATOR . 'presta_theme' . DIRECTORY_SEPARATOR . 'assets' . DIRECTORY_SEPARATOR . 'css' . DIRECTORY_SEPARATOR . 'var.css';

        // Générer le contenu du fichier var.css avec les nouvelles variables
        $css = ':root{
            --primary: ' . ($config->configGeneral[0]->primary ?? '') . ';
            --secondary: ' . ($config->configGeneral[0]->secondary ?? '') . ';
            --PoliceTitle: "' . ($config->configGeneral[0]->fontTitle ?? '') . '";
            --PoliceText: "' . ($config->configGeneral[0]->fontText ?? '') . '";
            --HeaderBackground: ' . ($config->configHeader[0]->background ?? '') . ';
            --HeaderColor: ' . ($config->configHeader[0]->color ?? '') . ';
            --FooterBackground: ' . ($config->configFooter[0]->background ?? '') . ';
            --FooterColor: ' . ($config->configFooter[0]->color ?? '') . ';
            --BandeauBackground: ' . ($config->configTopBandeauParameter[0]->background ?? '') . ';
            --BandeauColor: ' . ($config->configTopBandeauParameter[0]->color ?? '') . ';
        };';

        
        
        // Sauvegarder le nouveau contenu dans le fichier var.css
        file_put_contents($fileVarCss, $css);
    }
    
 
}