<?php
namespace PrestaShop\Module\LCVPrestaConnector;

/**
 * Gestionnaire de catégories
 * 
 */
class CategoriesManager
{
    /**
     * Synchroniseur
     * 
     * @var Syncer
     */
    private Syncer $syncer;

    /**
     * Catégories dynamiques gérées
     */
    public array $dynCats;

    /**
     * Mapping des catégories dynamiques
     */
    private array $mapping;

    /**
     * Liste exhaustive des catégories Prestashop de destination de mappage
     */
    private array $mappedCats;

    /**
     * Cache des catégories Prestashop
     */
    private array|null $psCatsCache = null;

    /** 
     * Nouvelle instance
     * 
     * @param Syncer $syncer Synchroniseur
     */
    public function __construct(Syncer $syncer, ?array $productsId = null)
    {
        $this->syncer = $syncer;
        // Chargement des mappings, pour chaque catégorie dynamique
        $mapping = [];
        $cfg = $this->syncer->module->getCfg();

        $this->dynCats = $cfg->get(SyncConfiguration::CFG_DYN_CAT);

        foreach ($this->dynCats as $cat => $name)
        {
            $mapping[$cat] = $cfg->get(SyncConfiguration::CFG_MAP_DYN_CAT . $cat);
            if (!isset($mapping[$cat]))
                $mapping[$cat] = [];

            // On convertit les ID en int
            foreach ($mapping[$cat] as $code => $id_category)
            {
                $intmap = [];
                if (is_array($id_category))
                {
                    foreach ($id_category as $id)
                        $intmap[] = (int) $id;
                }
                else
                    $intmap[] = (int) $id_category;

                $mapping[$cat][$code] = $intmap;
            }
        }
        
        $this->mapping = $mapping;
        $this->mappedCats = [];
        foreach ($mapping as $cat => $map)
        {
            foreach ($map as $code => $id_category)
            {
                if (is_array($id_category))
                {
                    foreach ($id_category as $id)
                        $this->mappedCats[$id] = true;
                }
                else
                    $this->mappedCats[$id_category] = true;
            }
        }
    }

    /*
     * Détermine si l'id de la catégorie renseignée est le résultat
     * probable d'un mappage
     */
    public function isMappedCategory(int $id_category) : bool
    {
        return isset($this->mappedCats[$id_category]) && $this->mappedCats[$id_category];
    }

    /**
     * Traduit un code de catégorie en ID de catégories Prestashop selon le mapping prévu
     * 
     * @param string $category Catégorie dynamique
     * @param string $code Code
     * @return array<int>ID de catégories
     */
    public function translateCode($category, $code) : array
    {
        if (isset($this->mapping[$category]))
        {
            $id_category = 0;
        
            if (isset($this->mapping[$category][$code]))
                $id_category = $this->mapping[$category][$code];

            if ($id_category)
            {
                if (is_array($id_category))
                {
                    if (count($id_category) > 0 && (count($id_category) > 1 || $id_category[0] != 0))
                        return $id_category;
                    if ($code != '*')
                        return $this->translateCode($category, '*');  // catch-all !
                }
                else
                    return [$id_category];
            }
            else if ($code != '*')
                return $this->translateCode($category, '*');  // catch-all !            
        }

        return [];
    }    

    /**
     * Contrôle que les catégories passées en paramètre sont les mêmes que celles du produit
     * 
     * @param array<int> $cats Catégories à vérifier
     * @param array<int> $productCats Catégories du produit
     * @return bool Vrai si les catégories sont les mêmes     
     */
    public function isCatsTheSame(array $cats, array $productCats) : bool
    {
        // On compare que les tableaux sont équivalents (quelque soit l'ordre)        
        if (count($productCats) != count($cats))
            return false;

        foreach ($cats as $cat)
        {
            if (!in_array($cat, $productCats))
                return false;
        }

        foreach ($productCats as $cat)
        {
            if (!in_array($cat, $cats))
                return false;
        }

        return true;    // Ca a l'air ok !
    }

    /**
     * Obtient le niveau de profondeur d'une catégorie Prestashop ou 0 si la catégorie n'existe pas
     * 
     * @param int $id_category ID de la catégorie Prestashop
     * @return int Niveau de profondeur de la catégorie
     */
    public function getPsCategoryLevel(int $id_category) : int
    {
        // Cache non chargé, on le charge
        if ($this->psCatsCache === null)
        {
            $this->psCatsCache = [];
            $cats = \Category::getCategories(false, false, false);
            foreach ($cats as $cat)
                $this->psCatsCache[$cat['id_category']] = $cat['level_depth'];
        }

        return isset($this->psCatsCache[$id_category]) ? $this->psCatsCache[$id_category] : 0;
    }
}