<?php
namespace PrestaShop\Module\LCVPrestaConnector\Models;

/**
 * Objet d'interface contenant la définition d'un produit
 */
class Product extends SanitizedObject
{
    /**
     * Identifiant du produit sous le backoffice
     */
    public string $id;

    /**
     * Code du produit
     */
    public ?string $code;

    /**
     * ID d'attributs de produit
     * Un produit prestashop sera formé pour chaque nouvelle valeur composée de $id et de certains de ces attributs, selon la configuration.
     * Tous les attributs doivent être mentionné ici, même si ils ne sont pas utilisés pour la composition de l'identifiant, sous la forme
     * clé - valeur
     * 
     * Par exemple, pour séparer par couleur, le code couleur devra faire partie de ces attributs.
     * Les autres attributs seront utilisés pour les variantes.
     * 
     * @var array<string, AttributeRef> Attributs de produit
     */
    public array $productAttributes = [];

    /**
     * Nom du produit
     */
    public ?string $name = null;

    /**
     * Nom de la grille de tailles
     */
    public ?string $grilleTailles = null;

    /**
     * marque pour mappage
     * 
     * @var AttributeRef
     */
    public ?AttributeRef $manufacturer = null;

    /**
     * Description du produit
     */
    public ?string $description = null;

    /**
     * Largeur du produit, en mm
     */
    public ?int $width = null;

    /**
     * Hauteur du produit, en mm
     */
    public ?int $height = null;

    /**
     * profondeur du produit, en mm
     */
    public ?int $depth = null;

    /**
     * Poids du produit, en grammes
     */
    public ?int $weight = null;

    /**
     * Taux de TVA, exprimée en % (de 0 à 1)
     */
    public ?float $taxrate = null;

    /**
     * Eco taxe, exprimée en % (de 0 à 1)
     */
    public ?float $ecotax = null;  

    /**
     * Articles associés
     */
    public array $articles = [];

    /**
     * Photos du produit
     */
    public ?array $photos = null;

    /**
     * Stocks du produit
     */
    public array $stocks = [];

    /**
     * Nouveau produit
     * 
     * @param string $id Identifiant du produit
     * @param string $code Code du produit
     */
    public function __construct(
        string $id,
        string $code
        )
    {
        // Sanitize
        $this->id = $id;
        $this->code = $code;
    }

    /**
     * Nettoyage de la donnée sur les standards Prestashop
     */
    public function sanitize()
    {
        $this->id = $this->sanitizeStr($this->id, 32);
        $this->code = strtoupper($this->sanitizeStr($this->code, 32));
        $this->name = $this->sanitizeStr($this->name, 128);
        
        $this->manufacturer->sanitize();
        foreach ($this->productAttributes as $attribute)
            $attribute->sanitize();

        // Nettoyage des articles
        foreach ($this->articles as $article)
            $article->sanitize();

        // \n\r -> <br>
        $this->description = str_replace("\n", "<br>", $this->description);

        // On enlève les photos en doublons
        if ($this->photos)
            $this->photos = array_unique($this->photos);
    }
}