<?php
namespace PrestaShop\Module\LCVPrestaConnector;

/**
 * Gestionnaire de marques
 * C'est en réalité un abstract factory qui permet de traduire un code directement en le lisant depuis la configuration
 * pour éviter les transferts de mémoire
 * 
 */
class ManufacturersManager
{
    /**
     * Synchroniseur
     * 
     * @var Syncer
     */
    private Syncer $syncer;

    /** 
     * Nouvelle instance
     * 
     * @param Syncer $syncer Synchroniseur
     */
    public function __construct(Syncer $syncer)
    {
        $this->syncer = $syncer;
    }

    /**
     * Traduit un code de marque en id_manufacturer, ou null si non trouvé
     * 
     * @param string $code Code de marque
     * @return int|null ID de la marque
     */
    public function translateCode(string $code) : ?int
    {
        $id_manufacturer = $this->syncer->module->getCfg()->getMap(SyncConfiguration::CFG_MAP_MANUFACTURER, $code);
        if (!$id_manufacturer)  /* catch-all ? */
            $id_manufacturer = $this->syncer->module->getCfg()->getMap(SyncConfiguration::CFG_MAP_MANUFACTURER, "*");
            
        return $id_manufacturer;
    }

    /**
     * Obtient le nom d'une marque sur le backoffice en fonction de son code
     * 
     * @param string $code Code de la marque
     * @return string Nom de la marque
     */
    public function getManufacturerName(string $code) : string
    {
        return $this->syncer->module->getCfg()->getMap(SyncConfiguration::CFG_MAP_MANUFACTURER_NAME, $code) ?? '';
    }

    /**
     * Enregistre un nouvel id_manufacturer et name pour un code de marque
     * 
     * @param string $code Code de la marque
     * @param int $id ID de la marque
     * @param string $name Nom de la marque (falcultatif)
     */
    public function registerManufacturer(string $code, int $id, ?string $name = null)
    {
        // On le sauvegarde en base
        $cfg = $this->syncer->module->getCfg();
        $cfg->setMap(SyncConfiguration::CFG_MAP_MANUFACTURER, $code, $id);
        if ($name)
            $cfg->setMap(SyncConfiguration::CFG_MAP_MANUFACTURER_NAME, $code, $name);
    }
}